<?php
    /**
     * PortfolioController Class
     *
     * @package Wojo Framework
     * @author wojoscripts.com
     * @copyright 2023
     * @version 6.20: PortfolioController.php, v1.00 5/22/2023 10:29 AM Gewa Exp $
     *
     */
    
    namespace Wojo\Controller\Admin\Module\Portfolio;
    
    use Exception;
    use Wojo\Core\Content;
    use Wojo\Core\Controller;
    use Wojo\Core\Filter;
    use Wojo\Core\Logger;
    use Wojo\Core\Meta;
    use Wojo\Core\Router;
    use Wojo\Core\Session;
    use Wojo\Database\Paginator;
    use Wojo\Debug\Debug;
    use Wojo\Exception\FileNotFoundException;
    use Wojo\Exception\NotFoundException;
    use Wojo\File\File;
    use Wojo\Image\Image;
    use Wojo\Language\Language;
    use Wojo\Message\Message;
    use Wojo\Module\Portfolio\Portfolio;
    use Wojo\Url\Url;
    use Wojo\Utility\Utility;
    use Wojo\Validator\Validator;
    
    if (!defined('_WOJO')) {
        die('Direct access to this location is not allowed.');
    }
    
    class PortfolioController extends Controller
    {
        /**
         * @param $request
         * @param $response
         * @param $services
         */
        public function __construct($request, $response, $services)
        {
            parent::__construct($request, $response, $services);
        }
        
        /**
         * index
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function index(): void
        {
            $find = isset($_POST['find']) ? Validator::sanitize($_POST['find'], 'default', 30) : null;
            $lg = Language::$lang;
            
            if (isset($_GET['letter']) and $find) {
                $letter = Validator::sanitize($_GET['letter'], 'string', 2);
                $counter = $this->db->count(Portfolio::mTable, "WHERE title$lg LIKE '%" . trim($find) . "%' AND `title$lg` REGEXP '^$letter'")->run();
                $where = "WHERE `title$lg` LIKE '%" . trim($find) . "%' AND `title$lg` REGEXP '^$letter'";
                
            } elseif (isset($_POST['find'])) {
                $counter = $this->db->count(Portfolio::mTable, "WHERE `title$lg` LIKE '%" . trim($find) . "%'")->run();
                $where = "WHERE d.title$lg LIKE '%" . trim($find) . "%'";
            } elseif (isset($_GET['letter'])) {
                $letter = Validator::sanitize($_GET['letter'], 'string', 2);
                $where = "WHERE `title$lg` REGEXP '^$letter'";
                $counter = $this->db->count(Portfolio::mTable, "WHERE `title$lg` REGEXP '^$letter'")->run();
            } else {
                $counter = $this->db->count(Portfolio::mTable)->run();
                $where = null;
            }
            
            if (isset($_GET['order']) and count(explode('|', $_GET['order'])) == 2) {
                list($sort, $order) = explode('|', $_GET['order']);
                $sort = Validator::sanitize($sort, 'default', 16);
                $order = Validator::sanitize($order, 'default', 4);
                if (in_array($sort, array(
                    'title',
                    'client',
                    'category_id'
                ))) {
                    $ord = ($order == 'DESC') ? ' DESC' : ' ASC';
                    $sorting = $sort . $ord;
                } else {
                    $sorting = ' created DESC';
                }
            } else {
                $sorting = ' created DESC';
            }
            
            $pager = Paginator::instance();
            $pager->items_total = $counter;
            $pager->default_ipp = $this->core->perpage;
            $pager->path = Url::url(Router::$path, '?');
            $pager->paginate();
            
            $sql = "
            SELECT d.id, d.created, d.category_id, d.client, d.title$lg as title, d.thumb, c.name$lg as name
              FROM `" . Portfolio::mTable . '` as d
              LEFT JOIN `' . Portfolio::cTable . "` as c ON c.id = d.category_id
              $where
              ORDER BY $sorting " . $pager->limit;
            
            $this->view->data = $this->db->rawQuery($sql)->run();
            $this->view->pager = $pager;
            
            $this->view->crumbs = ['admin', 'modules', Language::$word->_MOD_PF_TITLE];
            $this->view->caption = Language::$word->_MOD_PF_TITLE;
            $this->view->title = Language::$word->_MOD_PF_TITLE;
            $this->view->subtitle = null;
            $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
        }
        
        /**
         * new
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function new(): void
        {
            Session::set('foliotoken', Utility::randNumbers(4));
            $this->view->langlist = $this->core->langlist;
            $this->view->categories = Portfolio::categoryTree();
            $this->view->custom_fields = Content::renderCustomFields(0, 'portfolio');
            
            $this->view->crumbs = ['admin', 'modules', 'portfolio', Language::$word->_MOD_PF_NEWITM];
            $this->view->caption = Language::$word->_MOD_PF_NEWITM;
            $this->view->title = Language::$word->_MOD_PF_NEWITM;
            $this->view->subtitle = [];
            $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
        }
        
        /**
         * edit
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function edit(): void
        {
            $this->view->crumbs = ['admin', 'modules', 'portfolio', Language::$word->_MOD_PF_SUB2];
            $this->view->title = Language::$word->_MOD_PF_SUB2;
            $this->view->caption = Language::$word->_MOD_PF_SUB2;
            $this->view->subtitle = null;
            
            if (!$row = $this->db->select(Portfolio::mTable)->where('id', $this->view->matches, '=')->first()->run()) {
                if (DEBUG) {
                    $this->view->error = 'Invalid ID ' . ($this->view->matches) . ' detected [' . __CLASS__ . ', ln.:' . __line__ . ']';
                } else {
                    $this->view->error = Language::$word->META_ERROR;
                }
                $this->view->render('error', 'view/admin/');
            } else {
                $this->view->data = $row;
                $this->view->langlist = $this->core->langlist;
                $this->view->categories = Portfolio::categoryTree();
                $this->view->custom_fields = Content::renderCustomFields($row->id, 'portfolio');
                $this->view->images = $this->db->select(Portfolio::gTable, array('id', 'name'))->where('parent_id', $row->id, '=')->orderBy('sorting', 'ASC')->run();
                
                $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
            }
        }
        
        /**
         * settings
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function settings(): void
        {
            $row = $this->_config();
            $this->view->data = $row->portfolio;
            
            $this->view->crumbs = ['admin', 'modules', 'portfolio', Language::$word->_MOD_PF_SUB4];
            $this->view->caption = Language::$word->_MOD_PF_SUB4;
            $this->view->title = Language::$word->_MOD_PF_SUB4;
            $this->view->subtitle = [];
            $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
        }
        
        /**
         * categoryNew
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function categoryNew(): void
        {
            $this->view->langlist = $this->core->langlist;
            $this->view->tree = Portfolio::categoryTree();
            $this->view->sortlist = Portfolio::getSortCategoryList($this->view->tree);
            
            $this->view->crumbs = ['admin', 'modules', 'blog', Language::$word->_MOD_PF_SUB3];
            $this->view->caption = Language::$word->_MOD_PF_SUB3;
            $this->view->title = Language::$word->_MOD_PF_SUB3;
            $this->view->subtitle = [];
            $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
        }
        
        /**
         * categoryEdit
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function categoryEdit(): void
        {
            $this->view->crumbs = ['admin', 'modules', 'portfolio', Language::$word->_MOD_PF_SUB3];
            $this->view->title = Language::$word->_MOD_PF_SUB3;
            $this->view->caption = Language::$word->_MOD_PF_UPDATECAT;
            $this->view->subtitle = null;
            
            if (!$row = $this->db->select(Portfolio::cTable)->where('id', $this->view->matches, '=')->first()->run()) {
                if (DEBUG) {
                    $this->view->error = 'Invalid ID ' . ($this->view->matches) . ' detected [' . __CLASS__ . ', ln.:' . __line__ . ']';
                } else {
                    $this->view->error = Language::$word->META_ERROR;
                }
                $this->view->render('error', 'view/admin/');
            } else {
                $this->view->data = $row;
                $this->view->langlist = $this->core->langlist;
                $this->view->tree = Portfolio::categoryTree();
                $this->view->sortlist = Portfolio::getSortCategoryList($this->view->tree);
                
                $this->view->render('index', 'view/admin/modules/portfolio/view/', true, 'view/admin/');
            }
        }
        
        /**
         * action
         *
         * @return void
         * @throws NotFoundException
         */
        public function action(): void
        {
            $postAction = Validator::post('action');
            if ($postAction) {
                if (IS_AJAX) {
                    switch ($postAction) {
                        case 'add':
                        case 'update':
                            IS_DEMO ? Message::msgReply(true, 'info', Language::$word->PROCESS_ERR_DEMO) : $this->process();
                            break;
                        
                        case 'configuration':
                            IS_DEMO ? Message::msgReply(true, 'info', Language::$word->PROCESS_ERR_DEMO) : $this->configuration();
                            break;
                        
                        case 'category':
                            IS_DEMO ? Message::msgReply(true, 'info', Language::$word->PROCESS_ERR_DEMO) : $this->processCategory();
                            break;
                        
                        case 'images':
                            IS_DEMO ? Message::msgReply(true, 'success', Language::$word->PROCESS_ERR_DEMO) : $this->images();
                            break;
                        
                        case 'sort':
                            IS_DEMO ? print '0' : $this->sort(Validator::post('type'));
                            break;
                        
                        case 'delete':
                            IS_DEMO ? Message::msgReply(true, 'success', Language::$word->PROCESS_ERR_DEMO) : $this->_delete(Validator::post('type'));
                            break;
                        
                        default:
                            Url::invalidMethod();
                            break;
                    }
                } else {
                    Url::invalidMethod();
                }
            }
        }
        
        /**
         * process
         *
         * @return void
         */
        public function process(): void
        {
            $validate = Validator::run($_POST);
            foreach ($this->core->langlist as $lang) {
                $validate
                    ->set('title_' . $lang->abbr, Language::$word->NAME . ' <span class="flag icon ' . $lang->abbr . '"></span>')->required()->string()->min_len(3)->max_len(80)
                    ->set('slug_' . $lang->abbr, Language::$word->PAG_SLUG)->string()
                    ->set('body_' . $lang->abbr, Language::$word->DESCRIPTION)->text('advanced')
                    ->set('keywords_' . $lang->abbr, Language::$word->METAKEYS)->string(true, true)
                    ->set('description_' . $lang->abbr, Language::$word->METADESC)->string(true, true);
            }
            
            $validate
                ->set('category_id', Language::$word->CATEGORY)->required()->numeric()
                ->set('is_featured', Language::$word->_MOD_PF_SUB11)->required()->numeric()
                ->set('created_submit', Language::$word->CREATED)->required()->date()
                ->set('client', Language::$word->PAG_NOHEAD)->string();
            
            (Filter::$id) ? $this->_update($validate) : $this->_add($validate);
        }
        
        /**
         * _add
         *
         * @param Validator $validate
         * @return void
         */
        public function _add(Validator $validate): void
        {
            
            if (!array_key_exists('thumb', $_FILES)) {
                Message::$msgs['thumb'] = Language::$word->MAINIMAGE;
            }
            $thumb = File::upload('thumb', Portfolio::MAXIMG, 'png,jpg,jpeg');
            $file = File::upload('file', Portfolio::MAXFILE, 'zip,pdf,rar,mp3');
            
            $safe = $validate->safe();
            
            Content::verifyCustomFields('portfolio');
            
            if (count(Message::$msgs) === 0) {
                $data_m = array();
                foreach ($this->core->langlist as $i => $lang) {
                    $slug[$i] = (strlen($safe->{'slug_' . $lang->abbr}) === 0) ? Url::doSeo($safe->{'title_' . $lang->abbr}) : Url::doSeo($safe->{'slug_' . $lang->abbr});
                    $data_m['title_' . $lang->abbr] = $safe->{'title_' . $lang->abbr};
                    $data_m['slug_' . $lang->abbr] = $slug[$i];
                    $data_m['keywords_' . $lang->abbr] = $safe->{'keywords_' . $lang->abbr};
                    $data_m['description_' . $lang->abbr] = $safe->{'description_' . $lang->abbr};
                    $data_m['body_' . $lang->abbr] = Url::in_url($_POST['body_' . $lang->abbr]);
                    
                    if (strlen($safe->{'keywords_' . $lang->abbr}) === 0 or strlen($safe->{'description_' . $lang->abbr}) === 0) {
                        Meta::instance($safe->{'body_' . $lang->abbr});
                        if (strlen($safe->{'keywords_' . $lang->abbr}) === 0) {
                            $data_m['keywords_' . $lang->abbr] = Meta::getKeywords();
                        }
                        if (strlen($safe->{'description_' . $lang->abbr}) === 0) {
                            $data_m['description_' . $lang->abbr] = Meta::metaText($safe->{'body_' . $lang->abbr});
                        }
                    }
                }
                $data_x = array(
                    'category_id' => $safe->category_id,
                    'is_featured' => $safe->is_featured,
                    'client' => $safe->client,
                    'created' => $safe->created_submit,
                );
                
                $temp_id = Session::get('foliotoken');
                File::makeDirectory(FMODPATH . Portfolio::PORTDATA . $temp_id . '/thumbs');
                
                //process thumb
                if (array_key_exists('thumb', $_FILES)) {
                    $config = $this->_config();
                    $thumb_path = FMODPATH . Portfolio::PORTDATA . $temp_id . '/';
                    $thumb_result = File::process($thumb, $thumb_path, false);
                    try {
                        $img = new Image($thumb_path . $thumb_result['fname']);
                        $img->bestFit($config->portfolio->thumb_w, $config->portfolio->thumb_h)->save($thumb_path . 'thumbs/' . $thumb_result['fname']);
                    } catch (Exception $e) {
                        Debug::addMessage('errors', '<i>Error</i>', $e->getMessage(), 'session');
                    }
                    $data_x['thumb'] = $thumb_result['fname'];
                }
                //process file
                if (array_key_exists('file', $_FILES)) {
                    $file_result = File::process($file, FMODPATH . Portfolio::FILEDATA, false);
                    $data_x['file'] = $file_result['fname'];
                }
                
                $last_id = $this->db->insert(Portfolio::mTable, array_merge($data_m, $data_x))->run();
                
                // Start Custom Fields
                $fl_array = Utility::array_key_exists_wildcard($_POST, 'custom_*', 'key-value');
                if ($fl_array) {
                    $fields = $this->db->select(Content::cfTable)->run();
                    $data_array = array();
                    foreach ($fields as $row) {
                        $data_array[] = array(
                            'portfolio_id' => $last_id,
                            'field_id' => $row->id,
                            'field_name' => $row->name,
                            'section' => 'portfolio',
                        );
                    }
                    $this->db->batch(Content::cfdTable, $data_array)->run();
                    
                    foreach ($fl_array as $key => $val) {
                        $cf_data['field_value'] = Validator::sanitize($val);
                        $this->db->update(Content::cfdTable, $cf_data)->where('portfolio_id', $last_id, '=')->where('field_name', str_replace('custom_', '', $key), '=')->run();
                    }
                }
                
                //process gallery
                if ($rows = $this->db->select(Portfolio::gTable, array('id', 'parent_id'))->where('parent_id', Session::get('foliotoken'), '=')->run()) {
                    $query = 'UPDATE `' . Portfolio::gTable . '` SET `parent_id` = CASE ';
                    $list = '';
                    foreach ($rows as $item) {
                        $query .= ' WHEN id = ' . $item->id . ' THEN ' . $last_id;
                        $list .= $item->id . ',';
                    }
                    $list = substr($list, 0, -1);
                    $query .= ' END WHERE id IN (' . $list . ')';
                    $this->db->rawQuery($query)->run();
                }
                
                //rename temp folder
                File::renameDirectory(FMODPATH . Portfolio::PORTDATA . $temp_id, FMODPATH . Portfolio::PORTDATA . $last_id);
                
                if ($last_id) {
                    $message = Message::formatSuccessMessage($data_m['title' . Language::$lang], Language::$word->_MOD_PF_ITM_ADDED_OK);
                    $json['type'] = 'success';
                    $json['title'] = Language::$word->SUCCESS;
                    $json['message'] = $message;
                    $json['redirect'] = Url::url('admin/modules', 'portfolio/');
                    Logger::writeLog($message);
                } else {
                    $json['type'] = 'alert';
                    $json['title'] = Language::$word->ALERT;
                    $json['message'] = Language::$word->NOPROCESS;
                }
                print json_encode($json);
            } else {
                Message::msgSingleStatus();
            }
        }
        
        /**
         * _update
         *
         * @param Validator $validate
         * @return void
         */
        public function _update(Validator $validate): void
        {
            $thumb = File::upload('thumb', Portfolio::MAXIMG, 'png,jpg,jpeg');
            $file = File::upload('file', Portfolio::MAXFILE, 'zip,pdf,rar,mp3');
            
            $safe = $validate->safe();
            
            Content::verifyCustomFields('portfolio');
            
            if (count(Message::$msgs) === 0) {
                $data_m = array();
                foreach ($this->core->langlist as $i => $lang) {
                    $slug[$i] = (strlen($safe->{'slug_' . $lang->abbr}) === 0) ? Url::doSeo($safe->{'title_' . $lang->abbr}) : Url::doSeo($safe->{'slug_' . $lang->abbr});
                    $data_m['title_' . $lang->abbr] = $safe->{'title_' . $lang->abbr};
                    $data_m['slug_' . $lang->abbr] = $slug[$i];
                    $data_m['keywords_' . $lang->abbr] = $safe->{'keywords_' . $lang->abbr};
                    $data_m['description_' . $lang->abbr] = $safe->{'description_' . $lang->abbr};
                    $data_m['body_' . $lang->abbr] = Url::in_url($_POST['body_' . $lang->abbr]);
                    
                    if (strlen($safe->{'keywords_' . $lang->abbr}) === 0 or strlen($safe->{'description_' . $lang->abbr}) === 0) {
                        Meta::instance($safe->{'body_' . $lang->abbr});
                        if (strlen($safe->{'keywords_' . $lang->abbr}) === 0) {
                            $data_m['keywords_' . $lang->abbr] = Meta::getKeywords();
                        }
                        if (strlen($safe->{'description_' . $lang->abbr}) === 0) {
                            $data_m['description_' . $lang->abbr] = Meta::metaText($safe->{'body_' . $lang->abbr});
                        }
                    }
                }
                $data_x = array(
                    'category_id' => $safe->category_id,
                    'is_featured' => $safe->is_featured,
                    'client' => $safe->client,
                    'created' => $safe->created_submit,
                    'images' => $this->db->select(Portfolio::gTable, array('name'))->where('parent_id', Filter::$id, '=')->run('json'),
                );
                
                //process thumb
                $row = $this->db->select(Portfolio::mTable, array('thumb', 'file'))->where('id', Filter::$id, '=')->first()->run();
                if (array_key_exists('thumb', $_FILES)) {
                    $thumb_path = FMODPATH . Portfolio::PORTDATA . Filter::$id . '/';
                    $thumb_result = File::process($thumb, $thumb_path, false);
                    $config = $this->_config();
                    
                    File::deleteFile($thumb_path . $row->thumb);
                    File::deleteFile($thumb_path . 'thumbs/' . $row->thumb);
                    try {
                        $img = new Image($thumb_path . $thumb_result['fname']);
                        $img->bestFit($config->portfolio->thumb_w, $config->portfolio->thumb_h)->save($thumb_path . 'thumbs/' . $thumb_result['fname']);
                    } catch (Exception $e) {
                        Debug::addMessage('errors', '<i>Error</i>', $e->getMessage(), 'session');
                    }
                    $data_x['thumb'] = $thumb_result['fname'];
                }
                //process file
                if (array_key_exists('file', $_FILES)) {
                    $file_result = File::process($file, FMODPATH . Portfolio::FILEDATA, false);
                    File::deleteFile(FMODPATH . Portfolio::FILEDATA . $row->file);
                    $data_x['file'] = $file_result['fname'];
                }
                
                $this->db->update(Portfolio::mTable, array_merge($data_m, $data_x))->where('id', Filter::$id, '=')->run();
                
                // Start Custom Fields
                $fl_array = Utility::array_key_exists_wildcard($_POST, 'custom_*', 'key-value');
                if ($fl_array) {
                    foreach ($fl_array as $key => $val) {
                        $cf_data['field_value'] = Validator::sanitize($val);
                        $this->db->update(Content::cfdTable, $cf_data)->where('portfolio_id', Filter::$id, '=')->where('field_name', str_replace('custom_', '', $key), '=')->run();
                    }
                }
                
                $message = Message::formatSuccessMessage($data_m['title' . Language::$lang], Language::$word->_MOD_PF_ITM_UPDATE_OK);
                Message::msgReply(true, 'success', $message);
                Logger::writeLog($message);
            } else {
                Message::msgSingleStatus();
            }
        }
        
        /**
         * configuration
         *
         * @return void
         */
        public function configuration(): void
        {
            $validate = Validator::run($_POST);
            $validate
                ->set('ipc', Language::$word->_MOD_PF_SUB5)->required()->numeric()
                ->set('fpp', Language::$word->_MOD_PF_SUB6)->required()->numeric()
                ->set('cols', Language::$word->_MOD_GA_COLS)->required()->numeric()
                ->set('show_cats', Language::$word->_MOD_PF_SUB8)->required()->numeric()
                ->set('social', Language::$word->_MOD_PF_SUB12)->required()->numeric()
                ->set('show_sort', Language::$word->_MOD_PF_SUB9)->required()->numeric()
                ->set('show_featured', Language::$word->_MOD_PF_SUB10)->required()->numeric()
                ->set('latest', Language::$word->_MOD_PF_SUB13)->required()->numeric()
                ->set('layout', Language::$word->_MOD_PF_SUB7)->required()->string()
                ->set('thumb_w', Language::$word->THUMB_W)->required()->numeric()->exact_len(3)
                ->set('thumb_h', Language::$word->THUMB_H)->required()->numeric()->exact_len(3);
            
            $safe = $validate->safe();
            
            if (count(Message::$msgs) === 0) {
                $data = array(
                    'portfolio' => array(
                        'ipc' => $safe->ipc,
                        'fpp' => $safe->fpp,
                        'cols' => $safe->cols,
                        'show_cats' => $safe->show_cats,
                        'social' => $safe->social,
                        'show_sort' => $safe->show_sort,
                        'show_featured' => $safe->show_featured,
                        'latest' => $safe->latest,
                        'layout' => $safe->layout,
                        'thumb_w' => $safe->thumb_w,
                        'thumb_h' => $safe->thumb_h,
                    )
                );
                
                Message::msgReply(File::writeToFile(AMODPATH . 'portfolio/config.json', json_encode($data, JSON_PRETTY_PRINT)), 'success', Language::$word->_MOD_PF_CUPDATED);
                Logger::writeLog(Language::$word->_MOD_PF_CUPDATED);
            } else {
                Message::msgSingleStatus();
            }
        }
        
        /**
         * processCategory
         *
         * @return void
         */
        public function processCategory(): void
        {
            $validate = Validator::run($_POST);
            foreach ($this->core->langlist as $lang) {
                $validate
                    ->set('name_' . $lang->abbr, Language::$word->NAME . ' <span class="flag icon ' . $lang->abbr . '"></span>')->required()->string()->min_len(3)->max_len(80)
                    ->set('slug_' . $lang->abbr, Language::$word->ITEMSLUG)->string()
                    ->set('keywords_' . $lang->abbr, Language::$word->METAKEYS)->string(true, true)
                    ->set('description_' . $lang->abbr, Language::$word->METADESC)->string(true, true);
            }
            
            $safe = $validate->safe();
            
            if (count(Message::$msgs) === 0) {
                $data_m = array();
                foreach ($this->core->langlist as $i => $lang) {
                    $slug[$i] = (strlen($safe->{'slug_' . $lang->abbr}) === 0) ? Url::doSeo($safe->{'name_' . $lang->abbr}) : Url::doSeo($safe->{'slug_' . $lang->abbr});
                    $data_m['name_' . $lang->abbr] = $safe->{'name_' . $lang->abbr};
                    $data_m['slug_' . $lang->abbr] = $slug[$i];
                    $data_m['keywords_' . $lang->abbr] = $safe->{'keywords_' . $lang->abbr};
                    $data_m['description_' . $lang->abbr] = $safe->{'description_' . $lang->abbr};
                }
                $last_id = 0;
                (Filter::$id) ? $this->db->update(Portfolio::cTable, $data_m)->where('id', Filter::$id, '=')->run() : $last_id = $this->db->insert(Portfolio::cTable, $data_m)->run();
                if (Filter::$id) {
                    $message = Message::formatSuccessMessage($data_m['name' . Language::$lang], Language::$word->_MOD_PF_CAT_UPDATE_OK);
                    Message::msgReply($this->db->affected(), 'success', $message);
                    Logger::writeLog($message);
                } else {
                    if ($last_id) {
                        $message = Message::formatSuccessMessage($data_m['name' . Language::$lang], Language::$word->_MOD_PF_CAT_ADDED_OK);
                        $json['type'] = 'success';
                        $json['title'] = Language::$word->SUCCESS;
                        $json['message'] = $message;
                        $json['redirect'] = Url::url('admin/modules/portfolio/categories');
                        Logger::writeLog($message);
                    } else {
                        $json['type'] = 'alert';
                        $json['title'] = Language::$word->ALERT;
                        $json['message'] = Language::$word->NOPROCESS;
                    }
                    print json_encode($json);
                }
            } else {
                Message::msgSingleStatus();
            }
        }
        
        /**
         * images
         *
         * @return void
         */
        private function images(): void
        {
            $num_files = count($_FILES['images']['tmp_name']);
            $file_dir = FMODPATH . Portfolio::PORTDATA . Filter::$id;
            $config = $this->_config();
            $json['type'] = 'error';
            File::makeDirectory($file_dir . '/thumbs');
            
            for ($x = 0; $x < $num_files; $x++) {
                $image = $_FILES['images']['name'][$x];
                $newName = 'IMG_' . Utility::randomString(12);
                $ext = substr($image, strrpos($image, '.') + 1);
                $name = $newName . '.' . strtolower($ext);
                $full_name = $file_dir . '/' . $name;
                
                if (!move_uploaded_file($_FILES['images']['tmp_name'][$x], $full_name)) {
                    die(Message::msgSingleError(Language::$word->FU_ERROR13));
                }
                
                $img = new Image($file_dir . '/' . $name);
                
                if (!$img->getSourceWidth()) {
                    $json = array(
                        'title' => Language::$word->ERROR,
                        'message' => Message::$msgs['name'] = Language::$word->FU_ERROR7 //invalid image
                    );
                    print json_encode($json);
                    exit;
                }
                
                try {
                    $img->thumbnail($config->portfolio->thumb_w, $config->portfolio->thumb_h)->save($file_dir . '/thumbs/' . $name);
                    $last_id = $this->db->insert(Portfolio::gTable, array('parent_id' => Filter::$id, 'name' => $name))->run();
                    
                    $json['html'][$x] = '
                    <div class="columns" id="item_' . $last_id . '" data-id="' . $last_id . '">
                      <div class="wojo compact segment center-align">
                        <div class="handle"><i class="icon grip horizontal"></i></div>
                          <img src="' . Portfolio::hasThumb($name, Filter::$id) . '" alt="" class="wojo rounded image">
                          <a data-set=\'{"option":[{"delete": "delete","id":' . $last_id . ', "type":"image"}], "url":"modules/portfolio/action/", "action":"delete", "parent":"#item_' . $last_id . '"}\' class="wojo mini icon negative simple button data"><i class="icon x alt"></i></a>
                      </div>
                    </div>';
                } catch (Exception $e) {
                    echo 'Error: ' . $e->getMessage();
                }
            }
            $json['type'] = 'success';
            print json_encode($json);
        }
        
        /**
         * sort
         *
         * @param string $type
         * @return void
         */
        private function sort(string $type): void
        {
            $i = 0;
            $list = '';
            $query = $type == 'images' ? 'UPDATE `' . Portfolio::gTable . '` SET `sorting` = CASE ' : 'UPDATE `' . Portfolio::cTable . '` SET `sorting` = CASE ';
            
            foreach ($_POST['sorting'] as $item) {
                $i++;
                $query .= ' WHEN id = ' . $item . ' THEN ' . $i . ' ';
                $list .= $item . ',';
            }
            $list = substr($list, 0, -1);
            $query .= 'END WHERE id IN (' . $list . ')';
            $this->db->rawQuery($query)->run();
        }
        
        /**
         * _delete
         *
         * @param string $type
         * @return void
         */
        private function _delete(string $type): void
        {
            $title = Validator::post('title') ? Validator::sanitize($_POST['title']) : null;
            
            switch ($type) {
                case 'image':
                    if ($row = $this->db->select(Portfolio::gTable)->where('id', Filter::$id, '=')->first()->run()) {
                        File::deleteFile(FMODPATH . Portfolio::PORTDATA . $row->parent_id . '/' . $row->name);
                        File::deleteFile(FMODPATH . Portfolio::PORTDATA . $row->parent_id . '/thumbs/' . $row->name);
                        $this->db->delete(Portfolio::gTable)->where('id', Filter::$id, '=')->run();
                        $json = array(
                            'type' => 'success',
                            'title' => Language::$word->SUCCESS
                        );
                    } else {
                        $json['type'] = 'error';
                    }
                    print json_encode($json);
                    break;
                
                case 'category':
                    $this->db->delete(Portfolio::cTable)->where('id', Filter::$id, '=')->run();
                    
                    $message = str_replace('[NAME]', $title, Language::$word->_MOD_PF_CAT_DEL_OK);
                    Message::msgReply(true, 'success', $message);
                    Logger::writeLog($message);
                    break;
                
                default:
                    if ($this->db->delete(Portfolio::mTable)->where('id', Filter::$id, '=')->run()) {
                        $this->db->delete(Content::cfdTable)->where('portfolio_id', Filter::$id, '=')->run();
                        File::deleteRecursive(FMODPATH . Portfolio::PORTDATA . Filter::$id, true);
                    }
                    
                    $message = str_replace('[NAME]', $title, Language::$word->_MOD_PF_DEL_OK);
                    Message::msgReply(true, 'success', $message);
                    Logger::writeLog($message);
                    break;
            }
        }
        
        /**
         * _config
         *
         * @return mixed
         */
        private function _config(): mixed
        {
            return json_decode(File::loadFile(AMODPATH . 'portfolio/config.json'));
        }
    }
