<?php
    /**
     * PortfolioController Class
     *
     * @package Wojo Framework
     * @author wojoscripts.com
     * @copyright 2023
     * @version 6.20: PortfolioController.php, v1.00 6/15/2023 12:54 PM Gewa Exp $
     *
     */
    
    namespace Wojo\Controller\Front\Module\Portfolio;
    
    use Wojo\Core\Content;
    use Wojo\Core\Controller;
    use Wojo\Core\Module;
    use Wojo\Core\Plugin;
    use Wojo\Core\Router;
    use Wojo\Database\Paginator;
    use Wojo\Exception\FileNotFoundException;
    use Wojo\File\File;
    use Wojo\Language\Language;
    use Wojo\Module\Portfolio\Portfolio;
    use Wojo\Url\Url;
    use Wojo\Utility\Utility;
    use Wojo\Validator\Validator;
    
    if (!defined('_WOJO')) {
        die('Direct access to this location is not allowed.');
    }
    
    class PortfolioController extends Controller
    {
        /**
         * @param $request
         * @param $response
         * @param $services
         */
        public function __construct($request, $response, $services)
        {
            parent::__construct($request, $response, $services);
        }
        
        /**
         * index
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function index(): void
        {
            $lg = Language::$lang;
            
            $this->view->title = str_replace('[COMPANY]', $this->core->company, Language::$word->META_T31);
            $this->view->keywords = null;
            $this->view->description = null;
            $this->view->meta = null;
            $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), Language::$word->_MOD_PF_TITLE1];
            
            $mSql = array("title$lg as title", "info$lg", "keywords$lg", "description$lg");
            if (!$this->view->data = $this->db->select(Module::mTable, $mSql)->where('modalias', 'portfolio', '=')->first()->run()) {
                if (DEBUG) {
                    $this->view->error = 'Invalid portfolio module detected [' . __CLASS__ . ', ln.:' . __line__ . ']';
                    $this->view->render('error', 'view/front/themes/' . $this->core->theme . '/');
                } else {
                    $this->view->title = Language::$word->META_ERROR;
                    $this->view->render('404', 'view/front/themes/' . $this->core->theme . '/');
                }
            } else {
                if (isset($_GET['order']) and count(explode('|', $_GET['order'])) == 2) {
                    list($sort, $order) = explode('|', $_GET['order']);
                    $sort = Validator::sanitize($sort, 'default', 16);
                    $order = Validator::sanitize($order, 'default', 4);
                    if (in_array($sort, array('title', 'created', 'category_id'))) {
                        $ord = ($order == 'DESC') ? ' DESC' : ' ASC';
                        $sorting = $sort . $ord;
                    } else {
                        $sorting = ' created DESC';
                    }
                } else {
                    $sorting = ' created DESC';
                }
                
                $config = $this->_config();
                
                if ($config->portfolio->show_featured) {
                    $featured = 'WHERE d.is_featured = 1 ';
                    $counter = $this->db->count(Portfolio::mTable, 'WHERE is_featured = 1')->run();
                } else {
                    $featured = null;
                    $counter = $this->db->count(Portfolio::mTable)->run();
                }
                
                $pager = Paginator::instance();
                $pager->items_total = $counter;
                $pager->default_ipp = $config->portfolio->fpp;
                $pager->path = Url::url(Router::$path, '?');
                $pager->paginate();
                
                $sql = "
                SELECT d.id, d.created, d.title$lg as title, d.slug$lg as slug, d.thumb, c.slug$lg as cslug, c.name$lg as ctitle
                  FROM `" . Portfolio::mTable . '` as d
                  LEFT JOIN `' . Portfolio::cTable . "` as c ON c.id = d.category_id
                  $featured
                  ORDER BY $sorting " . $pager->limit;
                
                $this->view->title = Url::formatMeta($this->view->data->title, $this->core->company);
                $this->view->keywords = $this->view->data->{'keywords' . $lg};
                $this->view->description = $this->view->data->{'description' . $lg};
                $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), $this->view->data->title];
                
                $this->view->rows = $this->db->rawQuery($sql)->run();
                $this->view->plugins = Plugin::getModulePlugins('portfolio');
                $this->view->layout = Plugin::moduleLayout($this->view->plugins);
                $this->view->pager = $pager;
                $this->view->settings = $config->portfolio;
                $this->view->categories = Portfolio::categoryTree();
                
                $this->view->render('mod_index', 'view/front/themes/' . $this->core->theme . '/');
            }
        }
        
        /**
         * category
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function category(): void
        {
            $lg = Language::$lang;
            
            $this->view->title = str_replace('[COMPANY]', $this->core->company, Language::$word->META_T31);
            $this->view->keywords = null;
            $this->view->description = null;
            $this->view->meta = null;
            $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), Language::$word->_MOD_PF_TITLE1];
            
            if (!$this->view->row = $this->db->select(Portfolio::cTable)->where("slug$lg", $this->view->matches, '=')->first()->run()) {
                if (DEBUG) {
                    $this->view->error = 'Invalid gallery module detected [' . __CLASS__ . ', ln.:' . __line__ . ']';
                    $this->view->render('error', 'view/front/themes/' . $this->core->theme . '/');
                } else {
                    $this->view->title = Language::$word->META_ERROR;
                    $this->view->render('404', 'view/front/themes/' . $this->core->theme . '/');
                }
            } else {
                if (isset($_GET['order']) and count(explode('|', $_GET['order'])) == 2) {
                    list($sort, $order) = explode('|', $_GET['order']);
                    $sort = Validator::sanitize($sort, 'default', 16);
                    $order = Validator::sanitize($order, 'default', 4);
                    if (in_array($sort, array(
                        'title',
                        'client',
                        'category_id'
                    ))) {
                        $ord = ($order == 'DESC') ? ' DESC' : ' ASC';
                        $sorting = $sort . $ord;
                    } else {
                        $sorting = ' created DESC';
                    }
                } else {
                    $sorting = ' created DESC';
                }
                $config = $this->_config();
                
                $pager = Paginator::instance();
                $pager->items_total = $this->db->count(Portfolio::mTable, 'WHERE category_id = ' . $this->view->row->id)->run();
                $pager->default_ipp = $config->portfolio->ipc;
                $pager->path = Url::url(Router::$path, '?');
                $pager->paginate();
                
                $sql = "
                SELECT d.id, d.created, d.title$lg as title, d.slug$lg as slug, d.thumb
                  FROM `" . Portfolio::mTable . "` as d
                  WHERE category_id = ?
                  ORDER BY $sorting " . $pager->limit;
                
                $this->view->rows = $this->db->rawQuery($sql, array($this->view->row->id))->run();
                $this->view->data = $this->db->select(Module::mTable, array("title$lg as title", "info$lg", "keywords$lg", "description$lg"))->where('modalias', 'portfolio', '=')->first()->run();
                $this->view->plugins = Plugin::getModulePlugins('portfolio');
                $this->view->layout = Plugin::moduleLayout($this->view->plugins);
                $this->view->settings = $config->portfolio;
                $this->view->categories = Portfolio::categoryTree();
                $this->view->pager = $pager;
                
                $this->view->title = Url::formatMeta($this->view->row->{'name' . $lg}, $this->core->company);
                $this->view->keywords = $this->view->data->{'keywords' . $lg};
                $this->view->description = $this->view->data->{'description' . $lg};
                $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), array(0 => $this->view->data->title, 1 => $this->core->modname['portfolio']), $this->view->row->{'name' . $lg}];
                
                $this->view->render('mod_index', 'view/front/themes/' . $this->core->theme . '/');
            }
        }
        
        /**
         * render
         *
         * @return void
         * @throws FileNotFoundException
         */
        public function render(): void
        {
            $lg = Language::$lang;
            
            $this->view->title = str_replace('[COMPANY]', $this->core->company, Language::$word->META_T31);
            $this->view->keywords = null;
            $this->view->description = null;
            $this->view->meta = null;
            $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), Language::$word->_MOD_PF_TITLE1];
            
            if (!$this->view->row = $this->db->select(Portfolio::mTable)->where("slug$lg", $this->view->matches, '=')->first()->run()) {
                if (DEBUG) {
                    $this->view->error = 'Invalid gallery module detected [' . __CLASS__ . ', ln.:' . __line__ . ']';
                    $this->view->render('error', 'view/front/themes/' . $this->core->theme . '/');
                } else {
                    $this->view->title = Language::$word->META_ERROR;
                    $this->view->render('404', 'view/front/themes/' . $this->core->theme . '/');
                }
            } else {
                $this->view->data = $this->db->select(Module::mTable, array("title$lg as title", "info$lg", "keywords$lg", "description$lg"))->where('modalias', 'portfolio', '=')->first()->run();
                $this->view->plugins = Plugin::getModulePlugins('portfolio');
                $this->view->layout = Plugin::moduleLayout($this->view->plugins);
                $config = $this->_config();
                $this->view->settings = $config->portfolio;
                $this->view->custom_fields = Content::renderCustomFieldsFront($this->view->row->id, 'portfolio');
                $this->view->images = Utility::jSonToArray($this->view->row->images);
                
                $this->view->title = Url::formatMeta($this->view->row->{'title' . $lg}, $this->core->company);
                $this->view->keywords = $this->view->row->{'keywords' . $lg};
                $this->view->description = $this->view->row->{'description' . $lg};
                $this->view->crumbs = [array(0 => Language::$word->HOME, 1 => ''), array(0 => $this->view->data->title, 1 => $this->core->modname['portfolio']), $this->view->row->{'title' . $lg}];
                $this->view->meta = '
			    <meta property="og:type" content="article" />
			    <meta property="og:title" content="' . $this->view->row->{'title' . $lg} . '" />
			    <meta property="og:image" content="' . Portfolio::hasThumb($this->view->row->thumb, $this->view->row->id) . '" />
			    <meta property="og:description" content="' . $this->view->title . '" />
			    <meta property="og:url" content="' . Url::url('/' . $this->core->modname['portfolio'], $this->view->matches) . '" />
			    ';
                
                $this->view->render('mod_index', 'view/front/themes/' . $this->core->theme . '/');
            }
        }
        
        /**
         * action
         *
         * @return void
         */
        public function action(): void
        {
            $postAction = Validator::post('action');
            if ($postAction) {
                if (IS_AJAX) {
                    switch ($postAction) {
                        case 'search':
                            $this->search();
                            break;
                        
                        default:
                            Url::invalidMethod();
                            break;
                    }
                } else {
                    Url::invalidMethod();
                }
            } else {
                Url::invalidMethod();
            }
        }
        
        /**
         * search
         *
         * @return void
         */
        private function search(): void
        {
            $lg = Language::$lang;
            
            if (Validator::post('string')) {
                $string = Validator::sanitize($_POST['string'], 'string', 30);
                if (strlen($string) > 3) {
                    $sql = "
                    SELECT id, thumb, title$lg as title, slug$lg as slug
                      FROM `" . Portfolio::mTable . "`
                      WHERE MATCH (title$lg) AGAINST ('" . $string . "*' IN BOOLEAN MODE)
                      ORDER BY title$lg
                      LIMIT 10
                    ";
                    
                    $html = '';
                    if ($result = $this->db->rawQuery($sql)->run()) {
                        $html .= '<div class="wojo divided list">';
                        foreach ($result as $row) {
                            $link = Url::url($this->core->modname['portfolio'], $row->slug);
                            $html .= '<div class="item align-middle">';
                            $html .= '<div class="content auto">';
                            $html .= '<img class="wojo small image" src="' . Portfolio::hasThumb($row->thumb, $row->id) . '" alt="">';
                            $html .= '</div>';
                            $html .= '<div class="content padding-left">';
                            $html .= '<a href="' . $link . '">' . $row->title . '</a>';
                            $html .= '</div>';
                            $html .= '</div>';
                        }
                        $html .= '</div>';
                        $json = array(
                            'status' => 'success',
                            'html' => $html,
                        );
                    } else {
                        $json['status'] = 'error';
                    }
                    print json_encode($json);
                }
            }
        }
        
        /**
         * _config
         *
         * @return mixed
         */
        private function _config(): mixed
        {
            return json_decode(File::loadFile(AMODPATH . 'portfolio/config.json'));
        }
    }