<?php
    /**
     * Portfolio Class
     *
     * @package Wojo Framework
     * @author wojoscripts.com
     * @copyright 2023
     * @version 6.20: Portfolio.php, v1.00 5/9/2023 3:10 PM Gewa Exp $
     *
     */
    
    namespace Wojo\Module\Portfolio;
    
    use Wojo\Database\Database;
    use Wojo\File\File;
    use Wojo\Language\Language;
    use Wojo\Url\Url;
    use Wojo\Validator\Validator;
    
    if (!defined('_WOJO')) {
        die('Direct access to this location is not allowed.');
    }
    
    class Portfolio
    {
        const mTable = 'mod_portfolio';
        const cTable = 'mod_portfolio_categories';
        const gTable = 'mod_portfolio_gallery';
        
        const PORTDATA = 'portfolio/data/';
        const FILEDATA = 'portfolio/datafiles/';
        const MAXIMG = 5242880;
        const MAXFILE = 52428800;
        
        public int $fpp;
        public int $ipc;
        public int $cols;
        public string $layout;
        public int $show_cats;
        public int $show_sort;
        public int $show_featured;
        public int $thumb_w;
        public int $thumb_h;
        public int $social;
        public int $latest;
        
        /**
         * getSortCategoryList
         *
         * @param array $array
         * @return string
         */
        public static function getSortCategoryList(array $array): string
        {
            
            $icon = '<i class="icon negative trash"></i>';
            $html = "<ul class=\"wojo nestable list\">\n";
            if ($array) {
                foreach ($array as $row) {
                    $html .= '<li class="item" data-id="' . $row->id . '">'
                        . '<div class="content">'
                        . '<div class="handle"><i class="icon grip horizontal"></i></div>'
                        . '<div class="text"><a href="' . Url::url('admin/modules/portfolio/category', $row->id) . '">' . $row->{'name' . Language::$lang} . '</a></div>'
                        . '<div class="actions"><a class="data" data-set=\'{"option":[{"action": "delete","title": "' . Validator::sanitize($row->{'name' . Language::$lang}, 'chars') . '","id":' . $row->id . ', "type":"category"}],"action":"delete","parent":"li", "url":"modules/portfolio/action/"}\'>' . $icon . '</a></div> '
                        . '</div>';
                    $html .= "</li>\n";
                    
                }
            }
            
            $html .= "</ul>\n";
            
            return $html;
        }
        
        /**
         * categoryTree
         *
         * @return mixed
         */
        public static function categoryTree(): mixed
        {
            return Database::Go()->select(self::cTable, array('id', 'name' . Language::$lang, 'slug' . Language::$lang))->orderBy('sorting', 'ASC')->run();
        }
        
        /**
         * latestItems
         *
         * @return mixed
         */
        public static function latestItems(): mixed
        {
            $lg = Language::$lang;
            $settings = json_decode(File::loadFile(AMODPATH . 'portfolio/config.json'));
            
            $sql = "
            SELECT p.id, p.title$lg as title, p.slug$lg as slug, p.body$lg as body, p.thumb, p.created
              FROM `" . self::mTable . '` as p
              ORDER BY RAND()
              LIMIT 0, ' . $settings->portfolio->latest;
            
            return Database::Go()->rawQuery($sql)->run();
        }
        
        
        public static function simpleItems(): mixed
        {
            $lg = Language::$lang;
            $settings = json_decode(File::loadFile(AMODPATH . 'portfolio/config.json'));
            $featured = $settings->portfolio->show_featured ? 'WHERE d.is_featured = 1 ' : null;
            
            $sql = "
            SELECT d.id, d.created, d.title$lg as title, d.slug$lg as slug, d.thumb, c.slug$lg as cslug, c.name$lg as ctitle
              FROM `" . self::mTable . '` as d
              LEFT JOIN `' . self::cTable . "` as c ON c.id = d.category_id
              $featured
              ORDER BY d.created
              DESC LIMIT 0, " . $settings->portfolio->latest;

            return Database::Go()->rawQuery($sql)->run();
        }
        
        /**
         * searchResults
         *
         * @param string|null $keyword
         * @return mixed
         */
        public static function searchResults(string|null $keyword): mixed
        {
            $lg = Language::$lang;
            $keyword = Validator::sanitize($keyword, 'text', 20);
            $sql = "
            SELECT title$lg AS title, body$lg AS body, slug$lg AS slug
              FROM `" . self::mTable . "`
              WHERE MATCH (title$lg) AGAINST ('" . $keyword . "*' IN BOOLEAN MODE)
              OR MATCH (body$lg) AGAINST ('" . $keyword . "*' IN BOOLEAN MODE)
              ORDER BY created DESC
              LIMIT 10
          ";
            
            return Database::Go()->rawQuery($sql)->run();
        }
        
        /**
         * sitemap
         *
         * @return mixed
         */
        public static function sitemap(): mixed
        {
            $lg = Language::$lang;
            
            return Database::Go()->select(self::mTable, array("title$lg AS title", "slug$lg AS slug"))->orderBy('created', 'DESC')->run();
        }
        
        /**
         * hasThumb
         *
         * @param string|null $thumb
         * @param int $id
         * @return string
         */
        public static function hasThumb(string|null $thumb, int $id): string
        {
            return $thumb ? FMODULEURL . self::PORTDATA . $id . '/thumbs/' . $thumb : UPLOADURL . 'blank.jpg';
        }
        
        /**
         * hasImage
         *
         * @param string|null $image
         * @param int $id
         * @return string
         */
        public static function hasImage(string|null $image, int $id): string
        {
            return $image ? FMODULEURL . self::PORTDATA . $id . '/' . $image : UPLOADURL . 'blank.jpg';
        }
    }